<?php
/**
 * Plugin Name: Visual Composer KIT
 * Plugin URI:  http://vckit.themelego.com
 * Description: WPBakery Page Builder addons collection (Visual Composer)
 * Version:     2.0.4
 * Author:      ThemeLego
 * Author URI:  http://themelego.com
 * Donate link: http://vckit.themelego.com
 * License:     GPLv2
 * Text Domain: legocreative
 * Domain Path: /languages
 *
 * @link http://vckit.themelego.com
 *
 * @package Visual Composer KIT
 * @version 2.0.2
 */

/**
 * Copyright (c) 2017 ThemeLego (email : support@themelego.com)
 */

if ( ! defined( 'ABSPATH' ) ) { exit; // Exit if accessed directly.
}

define( 'VCKIT_PLUGIN', __FILE__ );
define( 'VCKIT_PLUGIN_DIR', untrailingslashit( dirname( VCKIT_PLUGIN ) ) );
define( 'VCKIT_PLUGIN_URL', plugins_url( '', __FILE__ ) );

/**
 * Main initiation class
 *
 * @since  1.0
 */
final class VCAddonsKIT {

  /**
   * Current version
   *
   * @var  string
   * @since  1.0.0
   */
  const VERSION = '2.0.4';

  /**
   * URL of plugin directory
   *
   * @var string
   * @since  1.0
   */
  protected $url = '';

  /**
   * Path of plugin directory
   *
   * @var string
   * @since  1.0
   */
  protected $path = '';

  /**
   * Plugin basename
   *
   * @var string
   * @since  1.0
   */
  protected $basename = '';

  /**
   * Detailed activation error messages
   *
   * @var array
   * @since  1.0
   */
  protected $activation_errors = array();

  /**
   * Singleton instance of plugin
   *
   * @var VCAddonsKIT
   * @since  1.0
   */
  protected static $single_instance = null;

  /**
   * Default template dir
   *
   * @var String
   * @since  1.0
   */
  protected $template_path;

  /**
   * Custom user templates dir
   *
   * @var String
   * @since  1.0
   */
  protected $custom_user_templates_dir = false;

  /**
   * Flag default_scripts_enqueued
   */
  public static $default_scripts_enqueued = false;

  /**
   * Shortcodes
   */
  protected $shortcodes;

  /**
   * Instance of Vckit Options
   *
   * @since0.1
   * @var Vckit_Options
   */
  protected $options;

  /**
   * Creates or returns an instance of this class.
   *
   * @since  1.0
   * @return VCAddonsKIT A single instance of this class.
   */
  public static function get_instance() {
    if ( null === self::$single_instance ) {
      self::$single_instance = new self();
    }

    return self::$single_instance;
  }

  /**
   * Sets up our plugin
   *
   * @since  1.0
   */
  protected function __construct() {
    $this->basename = plugin_basename( __FILE__ );
    $this->url      = plugin_dir_url( __FILE__ );
    $this->path     = plugin_dir_path( __FILE__ );

    $this->template_path = $this->path . 'templates/';
  }

  /**
   * Load the required dependencies for this plugin.
   *
   * Create an instance of the loader which will be used to register the hooks
   * with WordPress.
   *
   * @since    2.0.1
   * @access   private
   */
  private function load_dependencies() {
    // Include additional php files here.
    require_once VCKIT_PLUGIN_DIR . '/includes/vckit-core-functions.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/vckit-templates.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/metabox/metabox-types.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/metabox/interactive-grid.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/vc-param-types.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/vc-icons.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/settings.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/vckit-menu.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/vckit-submenu.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/vckit-submenu-pages.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/icomoon.php';

    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/base.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/fonts.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/row.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/interactive.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/icon-box.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/animated-block.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/google-map.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/pricing-box.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/button.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/before-after-image.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/onepage_navigator.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/checklist.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/separator.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/contact-form7.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/counter.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/circle-counter.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/typed-effect.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/countdown.php';
    if ( in_array(
      'testimonials-by-woothemes/woothemes-testimonials.php',
      apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) )
    ) {
      require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/testimonial.php';
    }

    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/grid.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/grid/samsung.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/grid/animate.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/grid/google.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/grid/magazine.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/grid/interactive.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/video.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/sharer.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/tilt-hover.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/heading-effect.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/card-flip.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/image-swap.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/image-banner.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/interactive-navigation.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/heading-gradient.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/interactive-notification.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/image-tilt-effect.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/carousel.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/carousel-item.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/lightbox.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/message-box.php';
    if ( in_array(
      'mailchimp-for-wp/mailchimp-for-wp.php',
      apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) )
    ) {
      require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/mailchimp.php';
    }
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/drop-caps.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/team-member.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/progress-bar.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/qr-code.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/workflow.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/workflow-item.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/gradient-background.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/icon-showcase.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/icon-showcase-item.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/instafeed.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/scroller-image.php';
    require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/image-slider.php';

    // Deprecated elements
    if ( $this->is_php_version_compatible() ) {
      require_once VCKIT_PLUGIN_DIR . '/includes/shortcodes/deprecated/instagram-feed.php';
    }
  }

  /**
   * Check if php version compatible
   *
   * @since 2.0.1
   */

  public function is_php_version_compatible() {
    $php_version = PHP_VERSION;
    $php_version_bool = (version_compare($php_version,  '5.3.0') >=  0) ? true : false;

    return $php_version_bool;
  }

  /**
   * Attach other plugin classes to the base plugin class.
   *
   * @since  1.0
   * @return void
   */
  public function plugin_classes() {
    $this->options = new Vckit_Options( $this );
    $menu = new Vckit_Menu();
    $submenu = new Vckit_Submenu( new Vckit_Submenu_Pages() );
    $param_types = new Vckit_VC_Param_Types( $this );
    $icons = new Vckit_VC_Icons( $this );

    // Attach other plugin classes to the base plugin class.
    $this->shortcodes['row'] = new Vckit_Shortcode_Row( $this );
    $this->shortcodes['animated_block'] = new Vckit_Shortcode_Animated_Block( $this );
    $this->shortcodes['before_after_image'] = new Vckit_Shortcode_Before_After_Image( $this );
    $this->shortcodes['button'] = new Vckit_Shortcode_Button( $this );
    $this->shortcodes['card-flip'] = new Vckit_Shortcode_Card_Flip( $this );
    $this->shortcodes['carousel'] = new Vckit_Shortcode_Carousel( $this );
    $this->shortcodes['carousel_item'] = new Vckit_Shortcode_Carousel_Item( $this );
    $this->shortcodes['circle_counter'] = new Vckit_Shortcode_Circle_Counter( $this );
    $this->shortcodes['contact_form7'] = new Vckit_Shortcode_Contact_Form7( $this );
    $this->shortcodes['countdown'] = new Vckit_Shortcode_Countdown( $this );
    $this->shortcodes['counter'] = new Vckit_Shortcode_counter( $this );
    $this->shortcodes['drop_caps'] = new Vckit_Shortcode_Drop_Caps( $this );
    $this->shortcodes['google_map'] = new Vckit_Shortcode_Google_Map( $this );
    $this->shortcodes['grid_animate'] = new Vckit_Shortcode_Grid_Animate( $this );
    $this->shortcodes['grid_google'] = new Vckit_Shortcode_Grid_Google( $this );
    $this->shortcodes['grid_interactive'] = new Vckit_Shortcode_Grid_Interactive( $this );
    $this->shortcodes['grid_magazine'] = new Vckit_Shortcode_Grid_Magazine( $this );
    $this->shortcodes['grid_samsung'] = new Vckit_Shortcode_Grid_Samsung( $this );
    $this->shortcodes['heading_effect'] = new Vckit_Shortcodes_Heading_Effect( $this );
    $this->shortcodes['heading_gradient'] = new Vckit_Shortcode_Heading_Gradient( $this );
    $this->shortcodes['icon_box'] = new Vckit_Shortcode_Icon_Box( $this );
    $this->shortcodes['image_banner'] = new Vckit_Shortcode_Image_Banner( $this );
    $this->shortcodes['image_slider'] = new Vckit_Shortcode_Image_Slider( $this );
    $this->shortcodes['image_swap'] = new Vckit_Shortcode_Image_Swap( $this );
    $this->shortcodes['image_tilt_effect'] = new Vckit_Shortcode_Image_Tilt_Effect( $this );
    $this->shortcodes['interactive'] = new Vckit_Shortcode_Interactive( $this );
    $this->shortcodes['interactive_navigation'] = new Vckit_Shortcode_Interactive_Navigation( $this );
    $this->shortcodes['interactive_notification'] = new Vckit_Shortcode_Interactive_Notification( $this );
    $this->shortcodes['lightbox'] = new Vckit_Shortcode_Lightbox( $this );
    $this->shortcodes['checklist'] = new Vckit_Shortcode_Checklist( $this );
    if ( in_array(
      'mailchimp-for-wp/mailchimp-for-wp.php',
      apply_filters( 'active_plugins', get_option( 'active_plugins' ) ) )
    ) {
      $this->shortcodes['mailchimp'] = new Vckit_Shortcode_Mailchimp( $this );
    }
    $this->shortcodes['message_box'] = new Vckit_Shortcode_Message_Box( $this );
    $this->shortcodes['onepage_navigator'] = new Vckit_Shortcode_Onepage_Navigator( $this );
    $this->shortcodes['pricing_box'] = new Vckit_Shortcode_Pricing_Box( $this );
    $this->shortcodes['progress_bar'] = new Vckit_Shortcode_Progress_Bar( $this );
    $this->shortcodes['separator'] = new Vckit_Shortcode_Separator( $this );
    $this->shortcodes['sharer'] = new Vckit_Shortcode_Sharer( $this );
    $this->shortcodes['team_member'] = new Vckit_Shortcode_Team_Member( $this );

    if ( class_exists( 'Vckit_Shortcode_Testimonial' ) ) {
      $this->shortcodes['testimonial'] = new Vckit_Shortcode_Testimonial( $this );
    }

    $this->shortcodes['tilt_hover'] = new Vckit_Shortcode_Tilt_Hover( $this );
    $this->shortcodes['typed_effect'] = new Vckit_Shortcode_Typed_Effect( $this );
    $this->shortcodes['video'] = new Vckit_Shortcode_Video( $this );
    $this->shortcodes['qr_code'] = new Vckit_Shortcode_QR_Code( $this );
    $this->shortcodes['workflow'] = new Vckit_Shortcode_Workflow( $this );
    $this->shortcodes['workflow_item'] = new Vckit_Shortcode_Workflow_Item( $this );
    $this->shortcodes['gradient_background'] = new Vckit_Shortcode_Gradient_Background( $this );
    $this->shortcodes['icon_showcase'] = new Vckit_Shortcode_Icon_Showcase( $this );
    $this->shortcodes['icon_showcase_item'] = new Vckit_Shortcode_Icon_Showcase_Item( $this );
    $this->shortcodes['instafeed'] = new Vckit_Shortcode_Instafeed( $this );
    $this->shortcodes['scroller_image'] = new Vckit_Shortcode_Scroller_Image( $this );

    // Deprecated
    if ( $this->is_php_version_compatible() ) {
      $this->shortcodes['instagram_feed'] = new Vckit_Shortcode_Instagram_Feed( $this );
    }
  } // END OF PLUGIN CLASSES FUNCTION

  /**
   * Add hooks and filters
   *
   * @since  1.0
   * @return void
   */
  public function hooks() {
    // Priority needs to be:
    // < 10 for CPT_Core,
    // < 5 for Taxonomy_Core,
    // 0 Widgets because widgets_init runs at init priority 1.
    add_action( 'init', array( $this, 'init' ), 0 );
  }

  /**
   * Init hooks
   *
   * @since  1.0
   * @return void
   */
  public function init() {
    // bail early if requirements aren't met
    if ( ! $this->check_requirements() ) {
      return;
    }

    add_action( 'admin_enqueue_scripts', array( &$this, 'adminCss' ) );
    add_action( 'wp_enqueue_scripts', array( &$this, 'enqueueDefaultStyle' ) );
    add_action( 'wp_enqueue_scripts', array( &$this, 'enqueueDefaultScripts' ) );

    // Allow upload SVG
    add_filter( 'upload_mimes', array( $this, 'cc_mime_types' ) );

    // load translated strings for plugin
    load_plugin_textdomain( 'legocreative', false, dirname( $this->basename ) . '/languages/' );

    // initialize plugin classes
    $this->load_dependencies();
    $this->plugin_classes();
    $this->registerMetabox();
    $this->notices();
  }

  /**
   * Allow svg upload
   * @since 2.0
   */
  public function cc_mime_types( $mimes ) {
    if ( ! isset( $mimes['svg'] ) ) {
      $mimes['svg'] = 'image/svg+xml';
    }
    return $mimes;
  }

  /**
   * Set user directory name.
   *
   * Directory name is the directory name vc should scan for custom shortcodes template.
   *
   * @since    1.0
   * @access   public
   *
   * @param $dir - path to shortcodes templates inside developers theme
   */
  public function setCustomUserShortcodesTemplateDir( $dir ) {
    preg_replace( '/\/$/', '', $dir );
    $this->custom_user_templates_dir = $dir;
  }

  /**
   * Get default directory where shortcodes templates area placed.
   *
   * @since  1.0
   * @access public
   *
   * @return string - path to default shortcodes
   */
  public function getDefaultShortcodesTemplatesDir() {
    return $this->template_path . 'shortcodes/';
  }

  /**
   *
   * Get shortcodes template dir.
   *
   * @since  1.0
   * @access public
   *
   * @param $template
   *
   * @return string
   */
  public function getShortcodesTemplateDir( $template ) {
    return false !== $this->custom_user_templates_dir ? $this->custom_user_templates_dir . '/' . $template : locate_template( 'vckit_templates/' . $template );
  }

  /**
   * Gets absolute path for file/directory in filesystem.
   *
   * @since  4.2
   * @access public
   *
   * @param $name - name of path dir
   * @param string $file - file name or directory inside path
   *
   * @return string
   */
  public function path( $name, $file = '' ) {
    $path = $this->template_path . $name . ( strlen( $file ) > 0 ? '/' . preg_replace( '/^\//', '', $file ) : '' );

    return apply_filters( 'vckit_path_filter', $path );
  }

  public function adminCss( $hook ) {
    wp_enqueue_style( 'vckit_admin_general_css', plugins_url( 'assets/css/admin-general.css', __FILE__ ) );

    if (
      'toplevel_page_vckit' == $hook
      || 'vckit_page_vckit-module' == $hook
      || 'vckit_page_vckit-about' == $hook
      || 'vckit_page_vckit-smoothscroll' == $hook
    ) {
      wp_enqueue_style( 'vckit_admin_css', plugins_url( 'assets/css/admin-options.css', __FILE__ ) );
      wp_enqueue_script( 'vckit_admin_js', plugins_url( 'assets/js/admin-js.js', __FILE__ ) );
    }
  }

  public function getShortcodes() {
    return $this->shortcodes;
  }

  public function loadPageTemplates( $vc_templates ) {
    $templates = array();
    foreach ( glob( $this->template_path . 'pages/*.php' ) as $page ) {
      $templates[] = include( $page );
    }

    return array_merge( $templates, $vc_templates );
  }

  /**
   * Check if the plugin meets requirements and
   * disable it if they are not present.
   *
   * @since  1.0
   * @return boolean result of meets_requirements
   */
  public function check_requirements() {
    // bail early if pluginmeets requirements
    if ( $this->meets_requirements() ) {
      return true;
    }

    // Add a dashboard notice.
    add_action( 'all_admin_notices', array( $this, 'requirements_not_met_notice' ) );

    // Deactivate our plugin.
    add_action( 'admin_init', array( $this, 'deactivate_me' ) );

    return false;
  }

  /**
   * Deactivates this plugin, hook this function on admin_init.
   *
   * @since  1.0
   * @return void
   */
  public function deactivate_me() {
    // We do a check for deactivate_plugins before calling it, to protect
    // any developers from accidentally calling it too early and breaking things.
    if ( function_exists( 'deactivate_plugins' ) ) {
      deactivate_plugins( $this->basename );
    }
  }

  /**
   * Check that all plugin requirements are met
   *
   * @since  1.0
   * @return boolean True if requirements are met.
   */
  public function meets_requirements() {
    // Do checks for required classes / functions
    // function_exists('') & class_exists('').
    // We have met all requirements.
    // Add detailed messages to $this->activation_errors array

    // Check if Visual Composer is installed
    if ( ! defined( 'WPB_VC_VERSION' ) ) {
      return false;
    }

    return true;
  }

  /**
   * Adds a notice to the dashboard if the plugin requirements are not met
   *
   * @since  1.0
   * @return void
   */
  public function requirements_not_met_notice() {
    // compile default message
    $default_message = sprintf(
      __( 'Visual Composer KIT is missing requirements and has been <a href="%s">deactivated</a>. Please make sure all requirements are available.', 'legocreative' ),
      admin_url( 'plugins.php' )
    );

    // default details to null
    $details = null;

    // add details if any exist
    if ( ! empty( $this->activation_errors ) && is_array( $this->activation_errors ) ) {
      $details = '<small>' . implode( '</small><br /><small>', $this->activation_errors ) . '</small>';
    }

    // output errors
    ?>
    <div id="message" class="error">
      <p><?php echo $default_message; // @codingStandardsIgnoreLine ?></p>
      <?php echo $details; // @codingStandardsIgnoreLine ?>
    </div>
    <?php

    if ( ! defined( 'WPB_VC_VERSION' ) ) {
      $plugin_data = get_plugin_data( __FILE__ );
      echo '
      <div class="error"><p>' .
      sprintf( __( '<strong>%s</strong> requires <strong><a href="https://codecanyon.net/item/visual-composer-page-builder-for-wordpress/242431?ref=themelego" target="_blank">Visual Composer</a></strong> plugin to be installed and activated on your site.', 'legocreative' ), $plugin_data['Name'] ) . // @codingStandardsIgnoreLine
      '</p></div>';
    }
  }

  /**
   * Magic getter for our object.
   *
   * @since  1.0
   * @param string $field Field to get.
   * @throws Exception Throws an exception if the field is invalid.
   * @return mixed
   */
  public function __get( $field ) {
    if ( isset( $this->shortcodes[ $field ] ) ) {
      return $this->shortcodes[ $field ];
    }

    switch ( $field ) {
      case 'version':
        return self::VERSION;
      case 'basename':
      case 'url':
      case 'path':
      case 'options':
        return $this->$field;
      default:
        throw new Exception( 'Invalid ' . __CLASS__ . ' property: ' . $field );
    }
  }

  /**
   * Get the plugin path.
   *
   * @access public
   * @since  1.0
   * @return string
   */
  public function plugin_path() {
    return untrailingslashit( plugin_dir_path( __FILE__ ) );
  } // End plugin_path()

  /**
   * Get asset dir
   *
   * @access public
   * @since  1.0
   * @return string
   */
  public function assets_dir() {
    return plugins_url( 'assets/', __FILE__ );
  }

  public function enqueueDefaultStyle() {
    wp_enqueue_style( 'vckit-css', vc_addons_kit_assets( 'css/vc_addons_kit.css' ), array(), $this->__get( 'version' ) );

    // Visual composer
    wp_enqueue_style( 'animate-css', vc_asset_url( 'lib/bower/animate-css/animate.min.css' ), array(), WPB_VC_VERSION, false );
  }

  public function enqueueDefaultScripts() {

    // Javascript
    // Maps
    $google_maps_key = vckit_get_option( 'google_maps_key', 'AIzaSyBip-8fRLCyicJqDgEPem4Si1r6ULnXtto' );

    $dependencies = array( 'jquery', 'vckit-libjs' );
    $lib_dependencies = array( 'jquery' );


    $sharer = vckit_get_option( 'vckit_sharer' ) == 1;
    if ( $sharer ) {
      $dependencies[] = 'mojs';
      wp_register_script( 'mojs', vc_addons_kit_assets( 'js/mo.min.js' ), array(), $this->__get( 'version' ), true );
    }

    // Libs
    wp_register_script( 'gmap3', vc_addons_kit_assets( 'js/gmap3.min.js' ), array(), $this->__get( 'version' ), true );

    // Vckit
    wp_register_script( 'vckit_google_map_js', '//maps.google.com/maps/api/js?key=' . $google_maps_key, array(), $this->__get( 'version' ), true );
    wp_register_script( 'vckit-libjs', vc_addons_kit_assets( 'js/lib.js' ), $lib_dependencies, $this->__get( 'version' ), true );
    wp_register_script( 'vckit-js', vc_addons_kit_assets( 'js/shortcode.js' ), $dependencies, $this->__get( 'version' ), true );
    wp_register_script( 'vckit-google-map-shortcode', vc_addons_kit_assets( 'js/google-map.js' ), array( 'jquery', 'vckit_google_map_js', 'gmap3' ), $this->__get( 'version' ), true );

    wp_register_script( 'vckit-particlesjs', vc_addons_kit_assets( 'js/particles.min.js' ), array(), $this->__get( 'version' ), true );

    $smoothscroll = vckit_get_option( 'smooth_scroll' ) == 1;
    if ( $smoothscroll ) {
      wp_register_script( 'vckit_smoothscroll', vc_addons_kit_assets( 'js/smooth-scroll.min.js' ), array(), $this->__get( 'version' ), true );
    }

    if ( false === self::$default_scripts_enqueued ) {
      if ( $sharer ) {
        wp_enqueue_script( 'mojs' );
      }

      if ( $smoothscroll ) {
        wp_enqueue_script( 'vckit_smoothscroll' );
        add_action( 'wp_head', array( $this, 'initSmoothScroll' ) );
      }

      wp_enqueue_script( 'waypoints' );
      wp_enqueue_script( 'vckit-libjs' );
      wp_enqueue_script( 'vckit-js' );

      self::$default_scripts_enqueued = true;
    }
  }

  /**
   * Init smoothscroll
   *
   * @since 2.0
   */
  public function initSmoothScroll() {

    $speed = vckit_get_option( 'smooth_scroll_speed', 480 );

    echo "<script type='text/javascript'>
      jQuery(document).ready(function($) {
        var scroll = new SmoothScroll('a[href*=\"#\"]', {
          ignore: '[data-scroll-ignore], [data-lightbox], .vckit-navigator a, .open-vckit-lightbox',
          header: null,
          speed: " . esc_html( $speed ) . ",
          offset: 0,
          easing: 'easeInOutCubic',
          customEasing: function (time) {},
          before: function () {},
          after: function () {}
        });
      });
    </script>";
  }

  public function registerMetabox() {
    add_action( 'cmb2_admin_init', 'vc_addons_kit_register_interactive_grid_metabox' );
  }

  /**
   * Wrapper function around cmb2_get_option
   * @since  0.1.0
   * @param  string $key     Options array key
   * @param  mixed  $default Optional default value
   * @return mixed           Option value
   */
  function getOption( $key = '', $default = null ) {
    if ( function_exists( 'cmb2_get_option' ) ) {
      // Use cmb2_get_option as it passes through some key filters.
      return cmb2_get_option( $this->options->key, $key, $default );
    }
    // Fallback to get_option if CMB2 is not loaded yet.
    $opts = get_option( $this->options->key, $key, $default );
    $val = $default;
    if ( 'all' == $key ) {
      $val = $opts;
    } elseif ( is_array( $opts ) && array_key_exists( $key, $opts ) && false !== $opts[ $key ] ) {
      $val = $opts[ $key ];
    }
    return $val;
  }

  /**
   * Notices
   * Activation instructions & CodeCanyon rating notices START
   *
   * For theme developers who want to include our plugin, they will need
   * to disable this section. This can be done by include this line
   * in their theme:
   *
   * defined( 'VCKIT_DISABLE_RATING_NOTICE' ) or define( 'VCKIT_DISABLE_RATING_NOTICE', true );
   *
   * @return void
   */
  public function notices() {
    if ( ! defined( 'VCKIT_DISABLE_RATING_NOTICE' ) ) {
      add_action( 'admin_notices', array( $this, 'remindRating' ) );
      add_action( 'wp_ajax_' . __CLASS__ . '-ask-rate', array( $this, 'ajaxRemindHandler' ) );
    }
  }

  /**
   * Activate the plugin
   *
   * @since  1.0
   * @return void
   */
  public function _activate() {
    // Make sure any rewrite functionality has been loaded.
    flush_rewrite_rules();

    delete_transient( __CLASS__ . '-activated' );
    set_transient( __CLASS__ . '-activated', time(), MINUTE_IN_SECONDS * 2 );

    if ( ! defined( 'VCKIT_DISABLE_RATING_NOTICE' ) ) {
      delete_transient( __CLASS__ . '-ask-rate' );
      set_transient( __CLASS__ . '-ask-rate', time(), DAY_IN_SECONDS * 4 );

      update_option( __CLASS__ . '-ask-rate-placeholder', 1 );
    }
  }

  /**
   * Deactivate the plugin
   * Uninstall routines should be in uninstall.php
   *
   * @since  1.0
   * @return void
   */
  public function _deactivate() {
    delete_transient( __CLASS__ . '-activated' );
    delete_transient( __CLASS__ . '-ask-rate' );
    delete_option( __CLASS__ . '-ask-rate-placeholder' );
  }

  /**
   * Ajax handler for when a button is clicked in the 'ask rating' notice
   *
   * @return  void
   * @since 1.0
   **/
  public function ajaxRemindHandler() {
    check_ajax_referer( __CLASS__, '_nonce' );

    if ( isset( $_POST['type'] ) ) {
      if ( 'remove' == $_POST['type'] ) {
        delete_option( __CLASS__ . '-ask-rate-placeholder' );
      } else { // remind
        set_transient( __CLASS__ . '-ask-rate', time(), DAY_IN_SECONDS );
      }
    }

    die();
  }

  /**
   * Displays the notice for reminding the user to rate our plugin
   *
   * @since   1.0
   *
   * @return  void
   **/
  public function remindRating() {
    if ( get_option( __CLASS__ . '-ask-rate-placeholder' ) === false ) {
      return;
    }
    if ( get_transient( __CLASS__ . '-ask-rate' ) ) {
      return;
    }

    $pluginData = get_plugin_data( __FILE__ );
    $nonce = wp_create_nonce( __CLASS__ );

    // @codingStandardsIgnoreStart
    echo '<div class="updated legocreative-ask-rating" style="border-left-color: #3498db">
      <p>
        <img src="' . esc_url( vc_addons_kit_assets( 'img/logo.png' ) ) . '" style="display: block; margin-bottom: 10px"/>
        <strong>' . sprintf( __( 'Enjoying %s?', $this->basename ), $pluginData['Name'] ) . '</strong><br>' .
        __( 'Help us out by rating our plugin 5 stars in CodeCanyon! This will allow us to create more awesome products and provide top notch customer support.', $this->basename ) . '<br>' .
        '<button data-href="http://codecanyon.net/downloads?utm_source=' .
        urlencode( $pluginData['Name'] ) . '&utm_medium=rate_notice" class="button button-primary" style="margin: 10px 10px 10px 0;">' .
        __( 'Rate us 5 stars in CodeCanyon :)', $this->basename ) .
        '</button>' .
        '<button class="button button-secondary remind" style="margin: 10px 10px 10px 0;">' . __( 'Remind me tomorrow', $this->basename ) . '</button>' .
        '<button class="button button-secondary nothanks" style="margin: 10px 0;">' . __( 'I&apos;ve already rated!', $this->basename ) . '</button>' .
        '<script>
        jQuery(document).ready(function($) {
          "use strict";

          $(".legocreative-ask-rating button").click(function() {
            if ( $(this).is(".button-primary") ) {
              var $this = $(this);

              var data = {
                "_nonce": "' . $nonce . '",
                "action": "' . __CLASS__ . '-ask-rate",
                "type": "remove"
              };

              $.post(ajaxurl, data, function(response) {
                $this.parents(".updated:eq(0)").fadeOut();
                window.open($this.attr("data-href"), "_blank");
              });

            } else if ( $(this).is(".remind") ) {
              var $this = $(this);

              var data = {
                "_nonce": "' . $nonce . '",
                "action": "' . __CLASS__ . '-ask-rate",
                "type": "remind"
              };

              $.post(ajaxurl, data, function(response) {
                $this.parents(".updated:eq(0)").fadeOut();
              });

            } else if ( $(this).is(".nothanks") ) {
              var $this = $(this);

              var data = {
                "_nonce": "' . $nonce . '",
                "action": "' . __CLASS__ . '-ask-rate",
                "type": "remove"
              };

              $.post(ajaxurl, data, function(response) {
                $this.parents(".updated:eq(0)").fadeOut();
              });
            }
            return false;
          });
        });
        </script>
      </p>
    </div>';
    // @codingStandardsIgnoreEnd
  }
}

// Kick it off.
$vckit_plugin = VCAddonsKIT::get_instance();

add_action( 'plugins_loaded', array( $vckit_plugin, 'hooks' ) );
register_activation_hook( __FILE__, array( $vckit_plugin, '_activate' ) );
register_deactivation_hook( __FILE__, array( $vckit_plugin, '_deactivate' ) );
