<?php
/**
 * VCKIT Addons Options
 *
 * @since 0.1
 * @package Visual Composer KIT
 */

/**
 * Vckit Options class.
 *
 * @since 0.1
 */
class Vckit_Options {
  /**
   * Parent plugin class
   *
   * @var    VCAddonsKIT
   * @since  0.1
   */
  protected $plugin = null;

  /**
   * Option key, and option page slug
   *
   * @var    string
   * @since  0.1
   */
  public $key = 'vckit_options';

  /**
   * Options page metabox id
   *
   * @var    string
   * @since  0.1
   */
  protected $metabox_id = 'vckit_options_metabox';

  /**
   * Options Page title
   *
   * @var    string
   * @since  0.1
   */
  protected $title = '';

  /**
   * Options Page hook
   * @var string
   */
  protected $options_page = '';

  /**
   * Constructor
   *
   * @since  0.1
   * @param  VCAddonsKIT $plugin Main plugin object.
   * @return void
   */
  public function __construct( $plugin ) {
    $this->plugin = $plugin;
    $this->hooks();

    $this->title = __( 'Addons KIT', 'legocreative' );
  }

  /**
   * Initiate our hooks
   *
   * @since  0.1
   * @return void
   */
  public function hooks() {
    add_action( 'admin_init', array( $this, 'admin_init' ) );
    add_action( 'cmb2_admin_init', array( $this, 'add_options_page_metabox' ) );
    add_action( 'cmb2_admin_init', array( $this, 'add_module_metabox' ) );
    add_action( 'cmb2_admin_init', array( $this, 'add_smooth_scroll_metabox' ) );
  }

  /**
   * Register our setting to WP
   *
   * @since  0.1
   * @return void
   */
  public function admin_init() {
    register_setting( $this->key, $this->key );
  }

  /**
   * Admin page markup. Mostly handled by CMB2
   *
   * @since  0.1
   * @return void
   */
  public function admin_page_display() {
    ?>
    <div class="wrap cmb2-options-page <?php echo esc_attr( $this->key ); ?>">
      <h2><?php echo esc_html( get_admin_page_title() ); ?></h2>
      <?php cmb2_metabox_form( $this->metabox_id, $this->key ); ?>
    </div>
    <?php
  }

  /**
   * Add custom fields to the options page.
   *
   * @since  0.1
   * @return void
   */
  public function add_options_page_metabox() {
    $types = vc_addons_kit()->__get( 'grid_interactive' )->getSupportedPostTypes();
    $posttypes = array();
    foreach ( $types as $type ) {
      $posttypes[ $type[0] ] = $type[1];
    }

    $cmb = new_cmb2_box( array(
      'id'         => 'vckit_general_metabox',
      'hookup'     => false,
      'cmb_styles' => false,
      'show_on'    => array(
        // These are important, don't remove.
        'key'   => 'options-page',
        'value' => array( $this->key ),
      ),
    ) );

    $cmb->add_field( array(
      'name'    => __( 'Google maps api key', 'legocreative' ),
      'desc'    => __( 'Google maps api v3 required key, Get <a target="_blank" href="https://developers.google.com/maps/documentation/javascript/get-api-key"> here </a>', 'legocreative' ),
      'id'      => 'google_maps_key', // no prefix needed
      'type'    => 'text',
      'default' => 'AIzaSyBip-8fRLCyicJqDgEPem4Si1r6ULnXtto',
    ) );

    $cmb->add_field( array(
      'name'    => __( 'Override contact form 7 template', 'legocreative' ),
      'id'      => 'override_contact_form7_template',
      'default' => 1,
      'type'    => 'switch',
      'desc'    => __( 'Enable this option, the default template of contact form 7 will be override by visual composer kit', 'legocreative' ),
    ) );

    // Disable metabox if grid is not enable
    if ( 0 !== vckit_get_option( 'vckit_grid_interactive' ) ) {
      $cmb->add_field( array(
        'name'    => __( 'Interactive grid apply for?', 'legocreative' ),
        'id'      => 'interactive_grid_post_types',
        'type'    => 'multicheck',
        'desc'    => __( 'Metabox for custom interactive grid will appear on the post types you selected.', 'legocreative' ),
        'options' => $posttypes,
        'select_all_button' => false,
      ) );
    }

    $cmb->add_field( array(
      'name'    => __( 'Enable lazyload image', 'legocreative' ),
      'id'      => 'enable_lazyload',
      'default' => 1,
      'type'    => 'switch',
      'desc'    => __( 'Enable this option, lazylaod image will enabled for interactive hover, interactive grid, image swap, before after image, instagram feed, image banner', 'legocreative' ),
    ) );
  }

  /**
   * Add custom fields to the options page.
   *
   * @since  2.0
   * @return void
   */
  public function add_module_metabox() {

    $cmb = new_cmb2_box( array(
      'id'         => 'vckit_module_metabox',
      'hookup'     => false,
      'cmb_styles' => false,
      'show_on'    => array(
        // These are important, don't remove.
        'key'   => 'options-page',
        'value' => array( 'vckit-module' ),
      ),
    ) );

    $shortcodes = vc_addons_kit()->getShortcodes();
    foreach ( $shortcodes as $shortcode ) {
      $cmb->add_field( array(
        'type'    => 'switch',
        'name'    => $shortcode->getName(),
        'desc'    => $shortcode->getDesc(),
        'id'      => $shortcode->getShortcode(), // no prefix needed
        'type'    => 'text',
        'type'    => 'switch',
        'default' => $shortcode->isEnabledByDefault() ? 1 : 0,
      ) );
    }
  }

  /**
   * Add custom fields to the options page.
   *
   * @since  2.0
   * @return void
   */
  public function add_smooth_scroll_metabox() {

    $cmb = new_cmb2_box( array(
      'id'         => 'vckit_smooth_scroll_metabox',
      'hookup'     => false,
      'cmb_styles' => false,
      'show_on'    => array(
        // These are important, don't remove.
        'key'   => 'options-page',
        'value' => array( 'vckit-smoothscroll' ),
      ),
    ) );

    $cmb->add_field( array(
      'name'    => __( 'Enable Smooth scroll', 'legocreative' ),
      'id'      => 'smooth_scroll',
      'default' => 0,
      'type'    => 'switch',
      'desc'    => __( 'Enable this option, allow smooth scroll on your page', 'legocreative' ),
    ) );

    $cmb->add_field( array(
      'name'    => __( 'Scroll speed', 'legocreative' ),
      'id'      => 'smooth_scroll_speed',
      'type'    => 'text',
      'default' => '480',
      'desc'    => __( 'The speed of the scrolling effect.', 'legocreative' ),
    ) );
  }
}
