<?php
if ( ! defined( 'ABSPATH' ) ) { exit; // Exit if accessed directly.
}

/**
 * VC KIT Shortcode Base class.
 *
 * @since 1.0
 * @package Visual Composer KIT
 */

abstract class Vckit_Shortcode_Base {
  /**
   * Parent plugin class
   *
   * @var VCAddonsKIT
   * @since  1.0
   */
  protected $plugin = null;

  /**
   * Shortcode shortcode
   *
   * @var String
   * @since  1.0
   */
  protected $shortcode = null;

  /**
   * Shortcode name
   *
   * @var String
   * @since  2.0
   */
  protected $name = null;

  /**
   * Shortcode description
   *
   * @var String
   * @since  2.0
   */
  protected $desc = null;

  /**
   * Shortcode default enabled
   *
   * @var String
   * @since  2.0
   */
  protected $default_enabled = true;

  /**
   * Shortcode shortcode prefix
   *
   * @var String
   * @since  1.0
   */
  protected $prefix = '';

  /**
   * Shortcode template
   *
   * @var String
   * @since  1.0
   */
  protected $html_template = null;

  /**
   * Constructor
   *
   * @since  1.0
   * @param  VCAddonsKIT $plugin Main plugin object.
   * @return void
   */
  public function __construct( $plugin ) {
    $this->plugin = $plugin;

    $this->hooks();
    $this->init();
  }

  /**
   * Initial Shortcode
   *
   * @since 2.0
   */
  public function init() {

    // Check shortcode enabled
    $enabled = vckit_get_option( $this->shortcode );
    $enabled = is_null( $enabled ) ? $this->default_enabled : $enabled;

    if ( $enabled ) {
      // Register CSS and JS
      add_action( 'template_redirect', array( $this, 'registerScripts' ) );
      // Maps
      add_action( 'vc_before_init', array( $this, 'maps' ) );

      if ( $this->shortcode && ! shortcode_exists( $this->shortcode ) ) {
        add_shortcode( $this->shortcode, array( $this, 'renderShortcode' ) );
      }
    }
  }

  public function maps() {
    $params = $this->params();
    if ( count( $params ) ) {
      $params['base'] = $this->shortcode;
      $params['name'] = $this->prefix . $this->name;
      $params['description'] = $this->desc;
      if ( ! isset( $params['category'] ) ) {
        $params['category'] = __( 'KIT Addons', 'legocreative' );
      }

      if ( ! isset( $params['icon'] ) ) {
        $params['icon'] = vc_addons_kit_assets( 'img/icon.png' );
      }

      if ( ! isset( $params['controls'] ) ) {
        $params['controls'] = 'full';
      }

      $params['params'][] = array(
        'type' => 'vckit_tutorial',
        'param_name' => 'tutorial',
        'std' => vckit_get_video_tutorial( $this->shortcode ),
      );

      vc_map( $params );
    }
  }

  /**
   * Register Scripts
   *
   * @since  1.0.0
   *
   * @return void
   */
  public function registerScripts() {}

  /**
   * Enqueue Scripts
   *
   * @since  1.0.0
   *
   * @return void
   */
  public function enqueueScripts() {}

  /**
   * Custom hooks for shortcode
   *
   * @since  1.0
   * @return void
   */

  public function hooks() {}

  /**
   * Get custom params for shortcode
   *
   * @since  1.0
   * @return {Array} Visual composer shortcode params
   */
  public function params() {
    return array();
  }

  /**
   * Get current shortcode
   *
   * @since  1.0
   * @return string
   */
  public function getShortcode() {
    return $this->shortcode;
  }

  /**
   * Get current shortcode name
   *
   * @since  2.0
   * @return string
   */
  public function getName() {
    return $this->name;
  }

  /**
   * Get current shortcode description
   *
   * @since  2.0
   * @return string
   */
  public function getDesc() {
    return $this->desc;
  }

  /**
   * Get current shortcode enabled by default
   *
   * @since  2.0
   * @return string
   */
  public function isEnabledByDefault() {
    return $this->default_enabled;
  }

  /**
   * Parse shortcode to custom class, custom css
   *
   * @since  1.0.0
   * @return string
   */
  public function getCustomCSS( $atts ) {
    return '';
  }

  /**
   * Render shortcode
   *
   * @since  1.0
   *
   * @param $atts
   * @param null $content
   * @param null $tag
   *
   * @return string
   */
  public function renderShortcode( $atts, $content = null, $tag = null ) {
    //vc_addons_kit()->enqueueDefaultScripts();
    $this->enqueueScripts();
    return $this->render( $atts, $content );
  }

  /**
   * Render shortcode
   *
   * @since  1.0
   *
   * @param $atts
   * @param null $content
   * @param null $tag
   *
   * @return string
   */
  public function render( $atts, $content = null, $tag = null ) {
    return $this->output( $atts, $content );
  }

  /**
   * @param $atts
   * @param null $content
   * @param string $base
   *
   * vc_filter: vc_shortcode_output - hook to override output of shortcode
   *
   * @return string
   */
  public function output( $atts, $content = null, $base = '' ) {
    $this->atts = $this->prepareAtts( $atts );
    $this->shortcode_content = $content;
    $output = '';
    $content = empty( $content ) && ! empty( $atts['content'] ) ? $atts['content'] : $content;
    if ( ( vc_is_inline() || vc_is_page_editable() ) && method_exists( $this, 'contentInline' ) ) {
      $output .= $this->contentInline( $this->atts, $content );
    } else {
      $custom_output = VC_SHORTCODE_CUSTOMIZE_PREFIX . $this->shortcode;
      $custom_output_before = VC_SHORTCODE_BEFORE_CUSTOMIZE_PREFIX . $this->shortcode; // before shortcode function hook
      $custom_output_after = VC_SHORTCODE_AFTER_CUSTOMIZE_PREFIX . $this->shortcode; // after shortcode function hook

      // Before shortcode
      if ( function_exists( $custom_output_before ) ) {
        $output .= $custom_output_before( $this->atts, $content );
      } else {
        $output .= $this->beforeShortcode( $this->atts, $content );
      }
      // Shortcode content
      if ( function_exists( $custom_output ) ) {
        $output .= $custom_output( $this->atts, $content );
      } else {
        $output .= $this->content( $this->atts, $content );
      }
      // After shortcode
      if ( function_exists( $custom_output_after ) ) {
        $output .= $custom_output_after( $this->atts, $content );
      } else {
        $output .= $this->afterShortcode( $this->atts, $content );
      }
    }
    // Filter for overriding outputs
    $output = apply_filters( 'vckit_shortcode_output', $output, $this, $this->atts );

    return $output;
  }

  /**
   * @param $template
   *
   * @return string
   */
  protected function setTemplate( $template ) {
    return $this->html_template = apply_filters( 'vckit_shortcode_set_template_' . $this->shortcode, $template );
  }

  /**
   * @return bool
   */
  protected function getTemplate() {
    if ( isset( $this->html_template ) ) {
      return $this->html_template;
    }

    return false;
  }

  /**
   * @param $atts
   * @param null $content
   *
   * @return mixed|void
   */
  protected function content( $atts, $content = null ) {
    $shortcode_content = $this->loadTemplate( $atts, $content );

    // Shortcode custom css
    // @since 2.0
    $shortcode_custom_css = $this->getCustomCSS( $atts );
    if ( ! empty( $shortcode_custom_css ) ) {
      $shortcode_custom_css = strip_tags( $shortcode_custom_css );
      $shortcode_content .= '<style type="text/css" data-type="vckit-shortcode-css" data-shortcode="' . $this->getShortCode() . '">';
      $shortcode_content .= $shortcode_custom_css;
      $shortcode_content .= '</style>';
    }

    return $shortcode_content;
  }

  /**
   * @param $atts
   * @param null $content
   *
   * vc_filter: vc_shortcode_content_filter - hook to edit template content
   * vc_filter: vc_shortcode_content_filter_after - hook after template is loaded to override output
   *
   * @return mixed|void
   */
  protected function loadTemplate( $atts, $content = null ) {
    $output = '';
    if ( ! is_null( $content ) ) {
      $content = apply_filters( 'vc_shortcode_content_filter', $content, $this->shortcode );
    }
    $this->findShortcodeTemplate();
    if ( $this->html_template ) {
      ob_start();
      include( $this->html_template );
      $output = ob_get_contents();
      ob_end_clean();
    } else {
      // XXX: trigger_error is not allowed on VIP Production.
      trigger_error( sprintf( __( 'Template file is missing for `%1$s` shortcode. Make sure you have `%1$s` file in your theme folder.', 'js_composer' ), $this->shortcode, 'wp-content/themes/your_theme/vc_templates/' . $this->shortcode . '.php' ) ); // @codingStandardsIgnoreLine
    }

    return apply_filters( 'vc_shortcode_content_filter_after', $output, $this->shortcode );
  }

  /**
   * @param $atts
   *
   * @return array
   */
  protected function prepareAtts( $atts ) {
    $return = array();
    if ( is_array( $atts ) ) {
      foreach ( $atts as $key => $val ) {
        $return[ $key ] = str_replace( array(
          '`{`',
          '`}`',
          '``',
        ), array( '[', ']', '"' ), $val );
      }
    }

    return $return;
  }

  /**
   * Creates html before shortcode html.
   *
   * @param $atts - shortcode attributes list
   * @param $content - shortcode content
   *
   * @return string - html which will be displayed before shortcode html.
   */
  public function beforeShortcode( $atts, $content ) {
    return '';
  }

  /**
   * Creates html before shortcode html.
   *
   * @param $atts - shortcode attributes list
   * @param $content - shortcode content
   *
   * @return string - html which will be displayed after shortcode html.
   */
  public function afterShortcode( $atts, $content ) {
    return '';
  }

  /**
   * @param $el_class
   *
   * @return string
   */
  public function getExtraClass( $el_class ) {
    $output = '';
    if ( '' !== $el_class ) {
      $output = ' ' . str_replace( '.', '', $el_class );
    }

    return $output;
  }

  /**
   * Get CSS animation
   * @param  $css_animation
   * @see    include/classes/shortcodes/shortcodes.php
   * @since  1.0
   *
   * @param  $css_animation
   * @return string
   */
  public function getCSSAnimation( $css_animation ) {
    $output = '';
    if ( '' !== $css_animation ) {

      // Visual composer effect
      if ( 'top-to-bottom' === $css_animation ||
           'bottom-to-top' === $css_animation ||
           'left-to-right' === $css_animation ||
           'right-to-left' === $css_animation ||
           'appear' === $css_animation
      ) {
        $output = ' wpb_animate_when_almost_visible wpb_' . $css_animation;
      } else {
        // AnimateCSS effect
        $output = ' wpb_animate_when_almost_visible vckit-anim vckit-anim-' . $css_animation;
      }
    }

    return $output;
  }

  /**
   * Find html template for shortcode output.
   */
  protected function findShortcodeTemplate() {
    // Check template path in shortcode's mapping settings
    if ( ! empty( $this->settings['html_template'] ) && is_file( $this->settings( 'html_template' ) ) ) {
      return $this->setTemplate( $this->settings['html_template'] );
    }

    // Check template in theme directory
    $user_template = vckit_shortcodes_theme_templates( $this->shortcode . '.php' );
    if ( is_file( $user_template ) ) {
      return $this->setTemplate( $user_template );
    }

    // Check default place
    $default_dir = vc_addons_kit()->getDefaultShortcodesTemplatesDir();
    if ( is_file( $default_dir . $this->shortcode . '.php' ) ) {
      return $this->setTemplate( $default_dir . $this->shortcode . '.php' );
    }

    return '';
  }

  /**
   * Get image src from attach_image
   * @param  image int
   *
   * @return string
   */
  protected function getImageSrc( $image, $size = 'full' ) {
    $image_src = '';
    if ( '' != $image ) {
      $image_attr = wp_get_attachment_image_src( $image, $size );
      if ( $image_attr ) {
        $image_src = $image_attr[0];
      }
    }

    if ( ! $image_src ) {
      $image_src = vc_asset_url( 'vc/no_image.png' );
    }
    return $image_src;
  }

  /**
   * Get image src from attach_image
   * @param  image int
   *
   * @return string
   */
  protected function getImageAlt( $image ) {
    $alt = '';
    if ( '' != $image ) {
      $alt = get_post_meta( $image, '_wp_attachment_image_alt', true);
    }

    return $alt;
  }

  /**
   * Get image src from attach_image
   * @param  image int
   *
   * @return string
   */
  protected function getImageSrcset( $image, $size = 'full' ) {
    $image_src = '';
    if ( '' != $image ) {
      $image_src = wp_get_attachment_image_srcset( $image, $size );
    }

    if ( ! $image_src ) {
      $image_src = vc_asset_url( 'vc/no_image.png' );
    }

    return $image_src;
  }

  /**
   * Correct vckit carousel by add common options
   *
   * @since 2.0
   */
  protected function addCarouselOptions( $dependency = array(), $group = null, $is_full = false ) {
    $arrow_styles = array(
      __( 'Arrow (Default)', 'legocreative' )    => 'arrow',
      __( 'Square arrows', 'legocreative' )      => 'square',
      __( 'Circle arrows ', 'legocreative' )     => 'circle',
      __( 'Half-circle arrows', 'legocreative' ) => 'half-circle',
    );

    $arrow_positions = array();
    $dot_positions = array();

    if ( $is_full ) {
      $arrow_styles = array_merge( $arrow_styles, array(
        __( 'Square', 'legocreative' ) => 'slide',
        __( 'Bar', 'legocreative' ) => 'imgbar',
        __( 'Circle pop', 'legocreative' ) => 'circlepop',
        __( 'Round', 'legocreative' ) => 'roundslide',
        __( 'Split', 'legocreative' ) => 'split',
        __( 'Reveal', 'legocreative' ) => 'reveal',
        __( 'Flip', 'legocreative' ) => 'flip',
        __( 'Circle', 'legocreative' ) => 'circleslide',
        __( 'Grow', 'legocreative' ) => 'growpop',
        __( 'Diamond', 'legocreative' ) => 'diamond',
        __( 'Fill', 'legocreative' ) => 'fillslide',
        //__( 'Fill path', 'legocreative' ) => 'fillpath',
      ) );

      $arrow_positions = array(
        array(
          'type' => 'dropdown',
          'heading' => __( 'Arrow position', 'legocreative' ),
          'param_name' => 'arrow_position',
          'value'      => array(
            __( 'Outside', 'legocreative' ) => 'outside',
            __( 'Inside', 'legocreative' ) => 'inside',
          ),
          'description' => __( 'Select arrow position', 'legocreative' ),
          'dependency'  => array(
            'element' => 'arrow_style',
            'value'   => array(
              'slide',
              'imgbar',
              'circlepop',
              'roundslide',
              'split',
              'reveal',
              'flip',
              'circleslide',
              'growpop',
              'diamond',
              'fillslide',
              //'fillpath',
            ),
          ),
          'edit_field_class' => 'vc_col-sm-4',
        ),
        array(
          'type'        => 'checkbox',
          'heading'     => __( 'Hide arrow when mouse out', 'legocreative' ),
          'param_name'  => 'hide_arrow_mouse',
          'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
          'std'         => 'yes',
          'dependency'  => array(
            'element' => 'arrow_style',
            'value'   => array(
              'slide',
              'imgbar',
              'circlepop',
              'roundslide',
              'split',
              'reveal',
              'flip',
              'circleslide',
              'growpop',
              'diamond',
              'fillslide',
              //'fillpath',
            ),
          ),
          'edit_field_class' => 'vc_col-sm-4',
        ),
      );
      $dot_positions = array(
        array(
          'type' => 'dropdown',
          'heading' => __( 'Dots position', 'legocreative' ),
          'param_name' => 'dot_position',
          'value'      => array(
            __( 'Outside', 'legocreative' ) => 'outside',
            __( 'Inside', 'legocreative' ) => 'inside',
          ),
          'dependency'  => array(
            'element' => 'show_dot',
            'value'   => array( 'yes' ),
          ),
          'edit_field_class' => 'vc_col-sm-4',
        ),
      );

    }

    $params = array_merge( array(
      array(
        'type'        => 'checkbox',
        'heading'     => __( 'Show arrow', 'legocreative' ),
        'param_name'  => 'show_arrow',
        'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
        'description' => __( 'Enable this options to show Prev/Next Arrows ', 'legocreative' ),
        'std'         => 'yes',
      ),
      array(
        'type'       => 'dropdown',
        'heading'    => __( 'Arrow style', 'legocreative' ),
        'param_name' => 'arrow_style',
        'value'      => $arrow_styles,
        'std'         => 'arrow',
        'description' => __( 'Select the arrow style', 'legocreative' ),
        'dependency'  => array(
          'element' => 'show_arrow',
          'value'   => array( 'yes' ),
        ),
        'edit_field_class' => 'vc_col-sm-4',
      ),
    ), $arrow_positions, array(
      array(
        'type'        => 'checkbox',
        'heading'     => __( 'Show dot', 'legocreative' ),
        'param_name'  => 'show_dot',
        'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
        'description' => __( 'Enable this options to show the dot', 'legocreative' ),
      ),
    ), $dot_positions, array(
      array(
        'type'        => 'checkbox',
        'heading'     => __( 'Enable fade', 'legocreative' ),
        'param_name'  => 'fade',
        'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
        'description' => __( 'Enable fade instead of slide', 'legocreative' ),
      ),
      array(
        'type'        => 'checkbox',
        'heading'     => __( 'Infinite', 'legocreative' ),
        'param_name'  => 'infinite',
        'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
        'description' => __( 'Enable infinite', 'legocreative' ),
      ),
      array(
        'type'        => 'checkbox',
        'heading'     => __( 'Auto play', 'legocreative' ),
        'param_name'  => 'autoplay',
        'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
        'description' => __( 'Enables Autoplay', 'legocreative' ),
      ),
      array(
        'type'        => 'checkbox',
        'heading'     => __( 'Pause on hover', 'legocreative' ),
        'param_name'  => 'pause_on_hover',
        'value'       => array( __( 'Yes', 'legocreative' ) => 'yes' ),
        'description' => __( 'Pause Autoplay when a dot is hovered', 'legocreative' ),
        'dependency'  => array(
          'element' => 'autoplay',
          'value'   => array( 'yes' ),
        ),
      ),
      array(
        'type'        => 'textfield',
        'heading'     => __( 'Autoplay Speed', 'legocreative' ),
        'param_name'  => 'autoplay_speed',
        'value'       => '3000',
        'description' => __( 'Autoplay Speed in milliseconds', 'legocreative' ),
        'dependency'  => array(
          'element' => 'autoplay',
          'value'   => array( 'yes' ),
        ),
      ),
      array(
        'type'        => 'textfield',
        'heading'     => __( 'Slide Speed', 'legocreative' ),
        'param_name'  => 'speed',
        'value'       => '300',
        'description' => __( 'Slide/Fade animation speed in miliseconds', 'legocreative' ),
      ),
      array(
        'type' => 'textfield',
        'heading' => __( 'Desktop Columns Count' , 'legocreative' ),
        'param_name' => 'item_width_md',
        'std' => '1',
        'description' => __( 'Choose Desktop(PC Mode) Columns Count', 'legocreative' ),
        'edit_field_class' => 'vc_col-sm-4',
      ),
      array(
        'type' => 'textfield',
        'heading' => __( 'Tablet Columns Count' , 'legocreative' ),
        'param_name' => 'item_width_sm',
        'std' => '1',
        'description' => __( 'Choose Tablet Columns Count', 'legocreative' ),
        'edit_field_class' => 'vc_col-sm-4',
      ),
      array(
        'type' => 'textfield',
        'heading' => __( 'Mobile Columns Count' ,'legocreative' ),
        'param_name' => 'item_width_xs',
        'std' => '1',
        'description' => __( 'Choose Mobile Columns Count', 'legocreative' ),
        'edit_field_class' => 'vc_col-sm-4',
      ),
      array(
        'type'        => 'textfield',
        'heading'     => __( 'Slides to scroll', 'legocreative' ),
        'param_name'  => 'slides_to_scroll',
        'value'       => '1',
        'description' => __( 'Select slides to scroll', 'legocreative' ),
      ),
      array(
        'type' => 'dropdown',
        'heading' => __( 'Gap spacing', 'legocreative' ),
        'param_name' => 'gap',
        'value' => array(
          '0px' => '0',
          '1px' => '1',
          '2px' => '2',
          '3px' => '3',
          '4px' => '4',
          '5px' => '5',
          '10px' => '10',
          '15px' => '15',
          '20px' => '20',
          '25px' => '25',
          '30px' => '30',
        ),
        'std' => '0',
        'description' => __( 'Select gap between grid elements.', 'legocreative' ),
      ),
      array(
        'type'        => 'colorpicker',
        'heading'     => __( 'Dot color', 'legocreative' ),
        'param_name'  => 'dot_color',
        'dependency'  => array(
          'element' => 'show_dot',
          'value'   => array( 'yes' ),
        ),
        'edit_field_class' => 'vc_col-sm-6',
      ),
      array(
        'type'        => 'colorpicker',
        'heading'     => __( 'Dot active color', 'legocreative' ),
        'param_name'  => 'dot_active_color',
        'dependency'  => array(
          'element' => 'show_dot',
          'value'   => array( 'yes' ),
        ),
        'edit_field_class' => 'vc_col-sm-6',
      ),
      array(
        'type'        => 'colorpicker',
        'heading'     => __( 'Arrow color', 'legocreative' ),
        'param_name'  => 'arrow_color',
        'dependency'  => array(
          'element' => 'show_arrow',
          'value'   => array( 'yes' ),
        ),
        'edit_field_class' => 'vc_col-sm-6',
      ),
      array(
        'type'        => 'colorpicker',
        'heading'     => __( 'Arrow background color', 'legocreative' ),
        'param_name'  => 'arrow_bg',
        'dependency'  => array(
          'element' => 'show_arrow',
          'value'   => array( 'yes' ),
        ),
        'edit_field_class' => 'vc_col-sm-6',
      ),
    ) );

    if ( $group ) {
      foreach ( $params as $key => $param ) {
        $params[ $key ]['group'] = $group;
      }
    }

    return $params;
  }

  protected function addIconLibrary( $dependency = array(), $group = null ) {
    //$pixel_icons = vc_pixel_icons();
    $icomoon_icons = icomoon_icons();

    $params = array(
      array(
        'type'    => 'dropdown',
        'heading' => __( 'Icon library', 'legocreative' ),
        'value'   => array(
          __( 'Font Awesome', 'legocreative' ) => 'fontawesome',
          __( 'Open Iconic', 'legocreative' ) => 'openiconic',
          __( 'Typicons', 'legocreative' ) => 'typicons',
          __( 'Entypo', 'legocreative' ) => 'entypo',
          __( 'Linecons', 'legocreative' ) => 'linecons',
          //__( 'Pixel', 'legocreative' ) => 'pixelicons',
          __( 'Mono Social', 'legocreative' ) => 'monosocial',
          __( 'Icomoon Free', 'legocreative' ) => 'icomoon',
        ),
        'param_name'  => 'icon_type',
        'std'         => 'fontawesome',
        'description' => __( 'Select icon library.', 'legocreative' ),
        'dependency'  => $dependency,
      ),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_fontawesome',
        'value'      => 'fa fa-image',
        'settings'   => array(
          'emptyIcon' => false,
          // default true, display an "EMPTY" icon?
          // 'iconsPerPage' => 4000,
          // default 100, how many icons per/page to display
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value'   => 'fontawesome',
        ),
        'description' => __( 'Select icon from library.', 'legocreative' ),
      ),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_openiconic',
        'settings'   => array(
          'emptyIcon' => false,
          // default true, display an "EMPTY" icon?
          'type' => 'openiconic',
          // 'iconsPerPage' => 4000,
          // default 100, how many icons per/page to display
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value' => 'openiconic',
        ),
        'description' => __( 'Select icon from library.', 'legocreative' ),
      ),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_typicons',
        'settings'   => array(
          'emptyIcon' => false,
          // default true, display an "EMPTY" icon?
          'type' => 'typicons',
          // 'iconsPerPage' => 4000,
          // default 100, how many icons per/page to display
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value' => 'typicons',
        ),
        'description' => __( 'Select icon from library.', 'legocreative' ),
      ),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_entypo',
        'settings'   => array(
          'emptyIcon' => false,
          // default true, display an "EMPTY" icon?
          'type' => 'entypo',
          // 'iconsPerPage' => 4000,
          // default 100, how many icons per/page to display
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value' => 'entypo',
        ),
      ),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_linecons',
        'settings'   => array(
          'emptyIcon' => false,
          // default true, display an "EMPTY" icon?
          'type' => 'linecons',
          // 'iconsPerPage' => 4000,
          // default 100, how many icons per/page to display
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value' => 'linecons',
        ),
        'description' => __( 'Select icon from library.', 'legocreative' ),
      ),
      //array(
        //'type'       => 'iconpicker',
        //'heading'    => __( 'Icon', 'legocreative' ),
        //'param_name' => 'icon_pixelicons',
        //'settings'   => array(
          //'emptyIcon' => false,
          //// default true, display an "EMPTY" icon?
          //'type' => 'pixelicons',
          //'source' => $pixel_icons,
        //),
        //'dependency' => array(
          //'element' => 'icon_type',
          //'value' => 'pixelicons',
        //),
        //'description' => __( 'Select icon from library.', 'legocreative' ),
      //),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_monosocial',
        'value'      => 'vc-mono vc-mono-fivehundredpx', // default value to backend editor admin_label
        'settings'   => array(
          'emptyIcon' => false, // default true, display an "EMPTY" icon?
          'type' => 'monosocial',
          // 'iconsPerPage' => 4000, // default 100, how many icons per/page to display
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value' => 'monosocial',
        ),
        'description' => __( 'Select icon from library.', 'legocreative' ),
      ),
      array(
        'type'       => 'iconpicker',
        'heading'    => __( 'Icon', 'legocreative' ),
        'param_name' => 'icon_icomoon',
        'settings'   => array(
          'emptyIcon' => false,
          'type' => 'icomoon',
          'source' => $icomoon_icons,
        ),
        'dependency' => array(
          'element' => 'icon_type',
          'value' => 'icomoon',
        ),
        'description' => __( 'Select icon from library.', 'legocreative' ),
      ),
    );

    if ( $group ) {
      foreach ( $params as $key => $param ) {
        $params[ $key ]['group'] = $group;
      }
    }

    return $params;
  }

  public function getIconClass( $atts ) {
    extract( shortcode_atts( array(
      'icon_type'        => 'fontawesome',
      'icon_fontawesome' => '',
      'icon_openiconic'  => '',
      'icon_typicons'    => '',
      'icon_entypo'      => '',
      'icon_linecons'    => '',
      'icon_pixelicons'  => '',
      'icon_monosocial'  => 'vc-mono vc-mono-fivehundredpx',
      'icon_icomoon'  => 'icon icon-home',
    ), $atts ) );

    $defaultIconClass = 'fa fa-image';
    $iconClass = ! empty( ${'icon_' . $icon_type} ) ? ${'icon_' . $icon_type} : $defaultIconClass;

    // Enqueue needed font for icon element
    if ( 'pixelicons' !== $icon_type ) {
      vc_icon_element_fonts_enqueue( $icon_type );
    }

    return $iconClass;
  }

  public function getGoogleFonts( $google_fonts, $google_fonts_param_name ) {
    // You must have a param with name google_fonts
    $google_fonts_field = WPBMap::getParam( $this->shortcode, $google_fonts_param_name );
    $google_fonts_obj = new Vc_Google_Fonts();
    $google_fonts_field_settings = isset( $google_fonts_field['settings'], $google_fonts_field['settings']['fields'] ) ? $google_fonts_field['settings']['fields'] : array();
    $google_fonts_data = strlen( $google_fonts ) > 0 ? $google_fonts_obj->_vc_google_fonts_parse_attributes( $google_fonts_field_settings, $google_fonts ) : '';

    return $google_fonts_data;
  }

  public function getGoogleFontsStyle( $google_fonts_data ) {
    $styles = array();

    $google_fonts_family = explode( ':', $google_fonts_data['values']['font_family'] );
    $styles[] = 'font-family:' . $google_fonts_family[0];
    $google_fonts_styles = explode( ':', $google_fonts_data['values']['font_style'] );
    $styles[] = 'font-weight:' . $google_fonts_styles[1];
    $styles[] = 'font-style:' . $google_fonts_styles[2];

    return $styles;
  }

  public function isEnableLazyload() {
    return 0 !== vckit_get_option( 'enable_lazyload' );
  }

  function vcIconElementParams() {
    $icomoon_icons = icomoon_icons();

    return array(
      'name' => __( 'Icon', 'js_composer' ),
      'base' => 'vc_icon',
      'icon' => 'icon-wpb-vc_icon',
      'category' => __( 'Content', 'js_composer' ),
      'description' => __( 'Eye catching icons from libraries', 'js_composer' ),
      'params' => array(
        array(
          'type' => 'dropdown',
          'heading' => __( 'Icon library', 'js_composer' ),
          'value' => array(
            __( 'Font Awesome', 'js_composer' ) => 'fontawesome',
            __( 'Open Iconic', 'js_composer' ) => 'openiconic',
            __( 'Typicons', 'js_composer' ) => 'typicons',
            __( 'Entypo', 'js_composer' ) => 'entypo',
            __( 'Linecons', 'js_composer' ) => 'linecons',
            __( 'Mono Social', 'js_composer' ) => 'monosocial',
            __( 'Icomoon Free', 'legocreative' ) => 'icomoon',
          ),
          'admin_label' => true,
          'param_name' => 'type',
          'description' => __( 'Select icon library.', 'js_composer' ),
        ),
        array(
          'type' => 'iconpicker',
          'heading' => __( 'Icon', 'js_composer' ),
          'param_name' => 'icon_fontawesome',
          'value' => 'fa fa-adjust', // default value to backend editor admin_label
          'settings' => array(
            'emptyIcon' => false,
            // default true, display an "EMPTY" icon?
            'iconsPerPage' => 4000,
            // default 100, how many icons per/page to display, we use (big number) to display all icons in single page
          ),
          'dependency' => array(
            'element' => 'type',
            'value' => 'fontawesome',
          ),
          'description' => __( 'Select icon from library.', 'js_composer' ),
        ),
        array(
          'type' => 'iconpicker',
          'heading' => __( 'Icon', 'js_composer' ),
          'param_name' => 'icon_openiconic',
          'value' => 'vc-oi vc-oi-dial', // default value to backend editor admin_label
          'settings' => array(
            'emptyIcon' => false, // default true, display an "EMPTY" icon?
            'type' => 'openiconic',
            'iconsPerPage' => 4000, // default 100, how many icons per/page to display
          ),
          'dependency' => array(
            'element' => 'type',
            'value' => 'openiconic',
          ),
          'description' => __( 'Select icon from library.', 'js_composer' ),
        ),
        array(
          'type' => 'iconpicker',
          'heading' => __( 'Icon', 'js_composer' ),
          'param_name' => 'icon_typicons',
          'value' => 'typcn typcn-adjust-brightness', // default value to backend editor admin_label
          'settings' => array(
            'emptyIcon' => false, // default true, display an "EMPTY" icon?
            'type' => 'typicons',
            'iconsPerPage' => 4000, // default 100, how many icons per/page to display
          ),
          'dependency' => array(
            'element' => 'type',
            'value' => 'typicons',
          ),
          'description' => __( 'Select icon from library.', 'js_composer' ),
        ),
        array(
          'type' => 'iconpicker',
          'heading' => __( 'Icon', 'js_composer' ),
          'param_name' => 'icon_entypo',
          'value' => 'entypo-icon entypo-icon-note', // default value to backend editor admin_label
          'settings' => array(
            'emptyIcon' => false, // default true, display an "EMPTY" icon?
            'type' => 'entypo',
            'iconsPerPage' => 4000, // default 100, how many icons per/page to display
          ),
          'dependency' => array(
            'element' => 'type',
            'value' => 'entypo',
          ),
        ),
        array(
          'type' => 'iconpicker',
          'heading' => __( 'Icon', 'js_composer' ),
          'param_name' => 'icon_linecons',
          'value' => 'vc_li vc_li-heart', // default value to backend editor admin_label
          'settings' => array(
            'emptyIcon' => false, // default true, display an "EMPTY" icon?
            'type' => 'linecons',
            'iconsPerPage' => 4000, // default 100, how many icons per/page to display
          ),
          'dependency' => array(
            'element' => 'type',
            'value' => 'linecons',
          ),
          'description' => __( 'Select icon from library.', 'js_composer' ),
        ),
        array(
          'type' => 'iconpicker',
          'heading' => __( 'Icon', 'js_composer' ),
          'param_name' => 'icon_monosocial',
          'value' => 'vc-mono vc-mono-fivehundredpx', // default value to backend editor admin_label
          'settings' => array(
            'emptyIcon' => false, // default true, display an "EMPTY" icon?
            'type' => 'monosocial',
            'iconsPerPage' => 4000, // default 100, how many icons per/page to display
          ),
          'dependency' => array(
            'element' => 'type',
            'value' => 'monosocial',
          ),
          'description' => __( 'Select icon from library.', 'js_composer' ),
        ),
        array(
          'type'       => 'iconpicker',
          'heading'    => __( 'Icon', 'legocreative' ),
          'param_name' => 'icon_icomoon',
          'settings'   => array(
            'emptyIcon' => false,
            'type' => 'icomoon',
            'source' => $icomoon_icons,
          ),
          'dependency' => array(
            'element' => 'icon_type',
            'value' => 'icomoon',
          ),
          'description' => __( 'Select icon from library.', 'legocreative' ),
        ),
        array(
          'type' => 'dropdown',
          'heading' => __( 'Icon color', 'js_composer' ),
          'param_name' => 'color',
          'value' => array_merge( getVcShared( 'colors' ), array( __( 'Custom color', 'js_composer' ) => 'custom' ) ),
          'description' => __( 'Select icon color.', 'js_composer' ),
          'param_holder_class' => 'vc_colored-dropdown',
        ),
        array(
          'type' => 'colorpicker',
          'heading' => __( 'Custom color', 'js_composer' ),
          'param_name' => 'custom_color',
          'description' => __( 'Select custom icon color.', 'js_composer' ),
          'dependency' => array(
            'element' => 'color',
            'value' => 'custom',
          ),
        ),
        array(
          'type' => 'dropdown',
          'heading' => __( 'Background shape', 'js_composer' ),
          'param_name' => 'background_style',
          'value' => array(
            __( 'None', 'js_composer' ) => '',
            __( 'Circle', 'js_composer' ) => 'rounded',
            __( 'Square', 'js_composer' ) => 'boxed',
            __( 'Diamond', 'js_composer' ) => 'diamond',
            __( 'Rounded', 'js_composer' ) => 'rounded-less',
            __( 'Outline Circle', 'js_composer' ) => 'rounded-outline',
            __( 'Outline Square', 'js_composer' ) => 'boxed-outline',
            __( 'Outline Rounded', 'js_composer' ) => 'rounded-less-outline',
          ),
          'description' => __( 'Select background shape and style for icon.', 'js_composer' ),
        ),
        array(
          'type' => 'dropdown',
          'heading' => __( 'Background color', 'js_composer' ),
          'param_name' => 'background_color',
          'value' => array_merge( getVcShared( 'colors' ), array( __( 'Custom color', 'js_composer' ) => 'custom' ) ),
          'std' => 'grey',
          'description' => __( 'Select background color for icon.', 'js_composer' ),
          'param_holder_class' => 'vc_colored-dropdown',
          'dependency' => array(
            'element' => 'background_style',
            'not_empty' => true,
          ),
        ),
        array(
          'type' => 'colorpicker',
          'heading' => __( 'Custom background color', 'js_composer' ),
          'param_name' => 'custom_background_color',
          'description' => __( 'Select custom icon background color.', 'js_composer' ),
          'dependency' => array(
            'element' => 'background_color',
            'value' => 'custom',
          ),
        ),
        array(
          'type' => 'dropdown',
          'heading' => __( 'Size', 'js_composer' ),
          'param_name' => 'size',
          'value' => array_merge( getVcShared( 'sizes' ), array( 'Extra Large' => 'xl' ) ),
          'std' => 'md',
          'description' => __( 'Icon size.', 'js_composer' ),
        ),
        array(
          'type' => 'dropdown',
          'heading' => __( 'Icon alignment', 'js_composer' ),
          'param_name' => 'align',
          'value' => array(
            __( 'Left', 'js_composer' ) => 'left',
            __( 'Right', 'js_composer' ) => 'right',
            __( 'Center', 'js_composer' ) => 'center',
          ),
          'description' => __( 'Select icon alignment.', 'js_composer' ),
        ),
        array(
          'type' => 'vc_link',
          'heading' => __( 'URL (Link)', 'js_composer' ),
          'param_name' => 'link',
          'description' => __( 'Add link to icon.', 'js_composer' ),
        ),
        vc_map_add_css_animation(),
        array(
          'type' => 'textfield',
          'heading' => __( 'Extra class name', 'js_composer' ),
          'param_name' => 'el_class',
          'description' => __( 'Style particular content element differently - add a class name and refer to it in custom CSS.', 'js_composer' ),
        ),
        array(
          'type' => 'css_editor',
          'heading' => __( 'CSS box', 'js_composer' ),
          'param_name' => 'css',
          'group' => __( 'Design Options', 'js_composer' ),
        ),
      ),
      'js_view' => 'VcIconElementView_Backend',
    );
  }

  public function getVcIcon( $atts ) {

    if ( empty( $atts['i_type'] ) ) {
      $atts['i_type'] = 'fontawesome';
    }
    $data = vc_map_integrate_parse_atts( $this->shortcode, 'vc_icon', $atts, 'i_' );
    if ( $data ) {
      $icon = visual_composer()->getShortCode( 'vc_icon' );
      if ( is_object( $icon ) ) {
        return $icon->render( array_filter( $data ) );
      }
    }

    return '';
  }

  public function getCarouselAtts( $atts ) {
    $show_arrow = '';
    $arrow_style = '';
    $show_dot = '';
    $fade = '';
    $infinite = '';
    $autoplay = '';
    $autoplay_speed = '';
    $pause_on_hover = '';
    $speed = '';
    $item_width_md = '';
    $item_width_sm = '';
    $item_width_xs = '';
    $slides_to_scroll = '';
    $gap = '';
    $dot_color = '';
    $dot_active_color = '';
    $arrow_color = '';
    $arrow_bg = '';

    $atts = vc_map_get_attributes( $this->getShortcode(), $atts );
    extract( $atts );

    $carousel = array(
      'arrows'           => 'yes' == $show_arrow,
      'dots'             => 'yes' == $show_dot,
      'fade'             => 'yes' == $fade,
      'infinite'             => 'yes' == $infinite,
      'arrow_style'      => $arrow_style,
      'autoplay'         => $autoplay,
      'autoplay_speed'   => intval( $autoplay_speed ),
      'pause_on_hover'   => $pause_on_hover,
      'speed'            => intval( $speed ),
      'md'               => intval( $item_width_md ),
      'sm'               => intval( $item_width_sm ),
      'xs'               => intval( $item_width_xs ),
      'slides_to_scroll' => intval( $slides_to_scroll ),
    );

    $carousel_attr = '';
    $carousel_attr = 'data-vckit-carousel=\'' . json_encode( $carousel ) . '\'';
    $carousel_attr .= ' data-gutter="' . $gap . '"';

    // Force fontawesome
    vc_icon_element_fonts_enqueue( 'fontawesome' );

    return $carousel_attr;
  }

  public function getCarouselStyle( $atts, $id ) {
    $dot_color = '';
    $dot_active_color = '';
    $arrow_color = '';
    $arrow_bg = '';

    $atts = vc_map_get_attributes( $this->getShortcode(), $atts );
    extract( $atts );

    $id = '#' . $id;
    $css = '';

    if ( $dot_color ) {
      $css .= $id . ' .slick-dots button { background-color: ' . $dot_color . '; }';
    }
    if ( $dot_active_color ) {
      $css .= $id . ' .slick-dots .slick-active button { background-color: ' . $dot_active_color . '; }';
    }
    if ( $arrow_color ) {
      $css .= $id . ' .slick-arrow { color: ' . $arrow_color . '; }';
    }
    if ( $arrow_bg ) {
      $css .= $id . ' .slick-arrow:not([class*="arrow-"]) { background-color: ' . $arrow_bg . '; }';
    }

    return $css;
  }

  // Fix conflict with Ultimate_VC_Addons
  // Fatal error: Call to undefined method Vckit_Shortcode_Card_Flip::settings() in Ultimate_VC_Addons/modules/Ultimate_Parallax.php on line 28
  public static function settings() {}
}
